{------------------------------------------------------------------------------}
{ SibVRV
{ svGE Unit
{ Created 12.03.2001 by Vereshagin Roman Vladimirovich.
{ History:
{  VR - Unit Created
{------------------------------------------------------------------------------}
unit svGE_SoundDriverClass;
interface
const
  svSound_DefaultBufferSize   = 1024; { Standard mix chunk size in samples }
  svSound_MaxChannels         = 5;
type
  tvol = array[0..svSound_MaxChannels-1] of word;
  tv   = array[0..svSound_MaxChannels-1] of longint;
  tvd  = array[0..svSound_MaxChannels-1] of longword;

  psample        = ^tsample;
  pdatchain      = ^tdatchain;
  psoundchain    = ^tsoundchain;

  PsvSoundData   = ^TsvSoundData;
  PsvSoundObject = ^TsvSoundObject;

  tdatchain = record
             current : PsvSoundData;
             next    : pdatchain;
           end;

  tsoundchain = record
             current : PsvSoundObject;
             next    : psoundchain;
           end;

  pcardinfo = ^tcardinfo;
  tcardinfo = record
    driverstring   : string[63];
    cardstring     : string[63];
    monoplay : record
      max,pref : longword;
    end;
    stereoplay : record
      max,pref : longword;
    end;
    minfreq        : longword;
    monorecfreq,
    stereorecfreq  : longword;
    flags          : longword;
    version        : longword;
    wavetablesize  : longword;
  end;

  { Sound driver object - only for heritage! }
  PsvSoundDriver = ^TsvSoundDriver;
  TsvSoundDriver = object
    chain       : tsoundchain;
    initialized : boolean;
    data : record
      listcount   : longword;  { Number of registered tsound objects }
      inchannels,              { Number of input channels (to mix)   }
      outchannels : longword;  { Number of out channels (to card)    }
      samplesize  : longword;  { Size of the output samples in bytes }
      flags       : longword;
      freq        : longword;  { Output frequency                    }
      buffer      : pointer;   { 32 bit mix buffer selector          }
      bufsize     : longword;  { Buffer size in samples              }
      flag_indos  : boolean;
      intcount,
      lastcount   : longword;
    end;
    cap       : tcardinfo;

    constructor init(bufsize : word);
    destructor  done;virtual;

    function    PlayBack_Start(freq,flags : longword) : boolean;virtual;
    procedure   PlayBack_Stop;virtual;

    function    change : boolean;virtual;
    procedure   setmixmanual(active : boolean);virtual;

    procedure   mixdata;virtual;


    function    getdatapointer : pointer;virtual;
    function    getdatasize    : longword;virtual;

    { ONLY Internal usage: }
    procedure   recalcvol;virtual;
    function    get2dvol(chn,vol,posx,posy : longint) : word;virtual;

    function    outofspec(freq,flags : longword) : boolean;virtual;
    private
      actblock  : word;
  end;

  { Sound object }
  TsvSoundObject = object
    output      : PsvSoundDriver;
    initialized : boolean;
    bfsize      : word;
    dat : record
      mainfreq    : longword; { Output playback frequency    }
      size        : longword; { Buffer size in bytes         }
      playchnls   : longword; { Maximum channels at once     }
      samplesize  : longword; { Sample size in bytes         }
      samplecount : longword; { Runtime sample count value   }
      lastcount   : longword; { Last samplecount value       }
      listcount   : longint;  { Number of registered objects }
      remain      : longword; { Requestblock data remain     }
      vol         : tvol;     { Volume multiplier values     }
    end;

    chain : tdatchain;       { Registered object chain      }

    constructor init(var dest : TsvSoundDriver;channels : longword);
    destructor  done;virtual;

    procedure   setbuffersize(samples : longword);virtual;
    procedure   setchannels(channels : longword);virtual;
    procedure   setvolume(channel : word;volume : longword);virtual;
    function    getvolume(channel : word) : word;virtual;

    function    change : boolean;virtual;

    procedure   setinterpolation(active : boolean);
    procedure   setramping(active : boolean);

    { ONLY Internal usage: }
    procedure   requestblock(p : pointer;samples : longword);virtual;
    function    getv(chn,vol : longword) : longword;virtual;
    procedure   callproc;virtual;
    private
    busy : boolean;
  end;

  TsvSoundData = object
    data : record
      samplesize : longword;  { Sample size in samples             }
      flags      : longword;
      ofreq      : longword;  { Original playback frequency        }
      freq       : longword;  { Current playback frequency         }
      buffer     : pointer;   { Sound data selector                }
      pos        : longword;  { Current sample pos in sound data   }
      xinc       : longword;  { Position increase delta value      }
      subpos     : longword;
      vol        : tvol;      { Physical channel volumes           }
      v          : tv;        { Logical channel volumes (internal) }
      vuval      : longword;
      vucount    : longword;
      lastvu     : byte;
      ramppos,                { Position within volume ramp        }
      rampstart,              { Ramping start value                }
      rampfinal,              { Final ramp value                   }
      oldvol    : tv;         { old volume value                   }
      oldsc     : tvd;        { old sample count                   }
    end;
    father      : PsvSoundObject;
    initialized : boolean;

    constructor init(var dest : TsvSoundObject);
    destructor  done;virtual;

    function    getnumofsamples : longword;virtual;
    function    getsamplesize : longword;virtual;
    function    getsamplefrequency : longword;virtual;

    procedure   play;virtual;
    procedure   pause;virtual;
    procedure   stop;virtual;
    function    isplaying : boolean;virtual;
    function    ispaused : boolean;virtual;
    function    ismuted : boolean;virtual;
    procedure   mute(active : boolean);virtual;
    procedure   setvolume(channel,volume : word);virtual;
    function    getvolume(channel : word) : word;virtual;
    procedure   setsoundpos(lr,cs,volume : integer);

    procedure   setpos(pos : longword);virtual;
    function    getpos : longword;virtual;
    function    getsize : longword;virtual;

    procedure   setspeed(speed : real);virtual;
    procedure   setfrequency(freq : longword);virtual;
    procedure   setinterpolation(active : boolean);virtual;
    procedure   setramping(active : boolean);virtual;

    function    getvu : byte;
    { Internal usage }
    procedure   requestblock(p : pointer;samples : longword);virtual;
    function    requestblock_direct16(p : pointer;samples : longword) : longword;virtual;
    private
    busy : boolean;
  end;

  tsample = object(TsvSoundData)
    dat : record
      size           : longword;
      loopstart,stop : longword;
    end;
    constructor init(var dest : TsvSoundObject;var source : TsvSoundData);
    destructor  done;virtual;

    function    getnumofsamples : longword;virtual;

    procedure   setloop(strt,stp : longword);
    procedure   setpos(pos : longword);virtual;
    function    getpos : longword;virtual;
    function    getsize : longword;virtual;

    procedure   requestblock(p : pointer;samples : longword);virtual;
  end;

  { Channel object - use preferably only for MOD playback }
  tchannel = object(tsample)
    constructor init(var dest : TsvSoundObject);
    destructor  done;virtual;

    procedure   setsample(s : pointer;size,freq : longword);
  end;

const
  snd_command   = $0000000F;
  snd_loopdata  = $00F00000;
  snd_play      = $00000001;
  snd_pause     = $00000002;
  snd_rec       = $00000004;

  snd_mute      = $00000010;

  snd_default   = $00000000;
  snd_mono      = $00000000;
  snd_stereo    = $00000100;
  snd_signed    = $00000200; { Sound data is signed          }
  snd_nonsigned = $00000000; { Sound data is non-signed      }
  snd_8bit      = $00000000; { Sound data uses 8 bits        }
  snd_16bit     = $00000400; { Sound data uses 16 bits       }
  snd_24bit     = $00000800; { Sound data uses 24 bits       }
  snd_prologic  = $00004000; { Dolby Pro Logic playback      }
  snd_mixmanual = $00010000; { Do manual poll mixing         }

  snd_intpol    = $00001000; { use sample interpolation      }
  snd_ramping   = $00002000; { use volume ramping            }

  snd_loop      = $00100000; { Sample is looped              }
  snd_inloop    = $00200000; { Sample currently in a loop    }
  snd_inramp    = $00400000; { in ramping activity           }

  snd_auto      = $10000000;

implementation

{ Driver management variables and types }
{ --- }
type
  pdriverchain = ^tdriverchain;
  tdriverchain = record
             driver   : PsvSoundDriver;
             detect   : function : boolean;
             next     : pdriverchain;
             priority : word;
           end;

const
  rampsize = 64; { Should actually be somewhere else... }
  noloop   = $FFFFFFFF;

{ TSoundDriver object procedures }
{ --- }
constructor TsvSoundDriver.init(bufsize : word);
begin
  initialized := true;
end;

destructor TsvSoundDriver.done;
begin
  initialized := false;
end;

function TsvSoundDriver.PlayBack_Start(freq,flags : longword) : boolean;
begin
  data.freq  := freq;
  data.flags := flags;
  Result := true;
end;

procedure TsvSoundDriver.PlayBack_Stop;
begin
end;

procedure TsvSoundDriver.recalcvol;
var
  ch  : psoundchain;
  i,j : integer;

begin
  ch := chain.next;
  for i := 0 to data.listcount-1 do
  begin
    for j := 0 to data.inchannels do
      ch^.current^.setvolume(j,ch^.current^.dat.vol[j]);
    ch := ch^.next;
  end;
end;

function TsvSoundDriver.change : boolean;
begin
  change         := (data.intcount <> data.lastcount);
  data.lastcount := data.intcount;
end;

procedure TsvSoundDriver.setmixmanual(active : boolean);
begin
  if active then data.flags := data.flags or snd_mixmanual
  else data.flags := data.flags and not snd_mixmanual;
end;

function TsvSoundDriver.get2dvol(chn,vol,posx,posy : longint) : word;
{  Get volume of a channnel in a 2D-field

          X-AXIS

            256



Y
|
A    L       0        R
X  -256              256
I
S


             S
           -256

Position 0 is the listener's position
}
var
  val : longint;

begin
  if (posx < -256) then posx := -256;
  if (posx >  256) then posx :=  256;

  if (posy < -256) then posy := -256;
  if (posy >  256) then posy :=  256;

  { If only stereo, don't allow S signal to disappear }
  if (data.flags and snd_prologic = 0) then
  begin
    if (posy < -128) then posy := -128;
  end;

  case chn of
    0 : val := (((256-posx))*(256-abs(posy))) shr 8;
    1 : val := (((256+posx))*(256-abs(posy))) shr 8;
    2 : begin
          if posy >= 0 then val := 0
          else val := abs(posy)-abs(posx) shr 3;
          if val < 0 then val := 0;
        end;
    else val := 0;
  end;
  if (val > 256) then val := 256;
  get2dvol := val*vol shr 8;
end;

procedure TsvSoundDriver.mixdata;
var
  i       : longint;
  ch      : psoundchain;
  p       : pointer;

begin
  actblock := (actblock+1) and 1;
  p := pointer(longword(data.buffer)+(longword(data.bufsize)*4*longword(data.inchannels))*actblock);
  { Reset mixing buffer to 0 }
  fillchar(data.buffer^,data.bufsize*data.inchannels*4*2,0);
  { Get mix data from all registered tsound objects }
  ch := chain.next;
  for i := 0 to longint(data.listcount)-1 do
  if (assigned(ch))and(assigned(ch^.current)) then
  begin
    ch^.current^.requestblock(p,data.bufsize);
    ch := ch^.next;
  end;
end;

function TsvSoundDriver.getdatapointer : pointer;
begin
  getdatapointer := pointer(longword(data.buffer)+(longword(data.bufsize)*4*longword(data.inchannels))*actblock);
end;

function TsvSoundDriver.getdatasize : longword;
begin
  getdatasize := data.bufsize;
end;

function TsvSoundDriver.outofspec(freq,flags : longword) : boolean;
begin
  outofspec := true;

  if (freq > cap.monoplay.max)and(cap.flags and snd_stereo = 0) then exit;
  if (flags and snd_stereo <> 0)and(cap.flags and snd_stereo = 0) then exit;
  if (flags and snd_24bit <> 0)and(cap.flags and snd_24bit = 0) then exit;
  if (flags and snd_16bit <> 0)and(cap.flags and snd_16bit = 0) then exit;
  if (flags and snd_stereo <> 0)and(freq > cap.stereoplay.max) then exit;
  if (flags and snd_stereo = 0)and(freq > cap.monoplay.max) then exit;
  outofspec := false;
end;
{ --- }


{ TSoundDat Object procedures }
{ --- }
constructor TsvSoundData.init(var dest : TsvSoundObject);
var
  i        : integer;
  newchain : pdatchain;

begin
  while busy do;
  busy := true;
  father := addr(dest);
  new(newchain);
  newchain^.next     := father^.chain.next;
  newchain^.current  := addr(self);
  father^.chain.next := newchain;
  inc(father^.dat.listcount);
  data.flags := data.flags or snd_ramping; { Ramping on by default }

  initialized := true;
  for i := 0 to 1 do setvolume(i,256);
  for i := 2 to svSound_MaxChannels-1 do setvolume(i,0);
  busy := false;
end;

destructor TsvSoundData.done;
var
  i      : longint;
  ch,ch2 : pdatchain;

begin
  while busy do;
  busy := true;
  ch  := father^.chain.next;
  ch2 := addr(father^.chain);
  for i := 0 to father^.dat.listcount-1 do
  begin
    if (ch^.current = addr(self)) then
    begin
      ch2^.next := ch^.next;
      dispose(ch);
      break;
    end;
    ch2 := ch;
    ch  := ch^.next;
  end;
  dec(father^.dat.listcount);
  busy := false;
  initialized := false;
end;

function TsvSoundData.getnumofsamples : longword;
begin
  getnumofsamples := 0;
end;

function TsvSoundData.getsamplesize : longword;
begin
  getsamplesize := data.samplesize;
end;

function TsvSoundData.getsamplefrequency : longword;
begin
  getsamplefrequency := data.ofreq;
end;

procedure TsvSoundData.setvolume(channel,volume : word);

begin
  if (volume > 512) then volume := 512;
  if (data.flags and snd_ramping <> 0) then with data do
  begin
    if (oldsc[channel] <> father^.dat.samplecount) then
    begin
      oldvol[channel] := v[channel];
      oldsc[channel] := father^.dat.samplecount;
    end;

    flags   := flags or snd_inramp;
    ramppos[channel] := 0;

    rampstart[channel] := oldvol[channel];
    rampfinal[channel] := father^.getv(channel,volume);
  end;
  data.vol[channel] := volume;
  data.v[channel]   := father^.getv(channel,volume);
end;

function TsvSoundData.getvolume(channel : word) : word;
begin
  if (channel > father^.output^.data.inchannels) then
    getvolume := 0 else getvolume := data.vol[channel];
end;

procedure TsvSoundData.setsoundpos(lr,cs,volume : integer);
var
  i : integer;

begin
  if not initialized then exit;
  if (volume > 512) then volume := 512;
  if (volume < 0) then volume := 0;
  for i := 0 to father^.output^.data.inchannels-1 do
    setvolume(i,father^.output^.get2dvol(i,volume,lr,cs));
end;

procedure TsvSoundData.setpos(pos : longword);
begin
end;

function TsvSoundData.getpos : longword;
begin
  getpos := 0;
end;

function TsvSoundData.getsize : longword;
begin
  getsize := 0;
end;


procedure TsvSoundData.play;
var
  i       : integer;

begin
  while busy do;
  busy := true;
  if (data.flags and snd_pause <> 0) then
    data.flags := (data.flags and not snd_pause) or snd_play
  else
  with data do
  begin
    pos    := 0;
    subpos := 0;
    flags  := flags or snd_play and not (snd_inloop);
  end;

  if (data.flags and snd_ramping <> 0) then
  with data do
  begin
    flags   := flags or snd_inramp;

    for i := 0 to svSound_MaxChannels-1 do
    begin
      oldvol[i] := 0;
      ramppos[i] := 0;
      rampstart[i] := 0;
      rampfinal[i] := v[i];
    end;
  end;
  busy := false;
end;

procedure TsvSoundData.pause;
begin
  while busy do;
  busy := true;
  if (data.flags and snd_pause <> 0) then
  begin
    data.flags := (data.flags and not snd_pause) or snd_play;
  end
  else if (data.flags and snd_play <> 0) then
  begin
    data.flags := (data.flags and not snd_play) or snd_pause;
  end;
  busy := false;
end;

procedure TsvSoundData.stop;
begin
  while busy do;
  busy := true;
  data.flags := data.flags and not (snd_play or snd_pause or snd_inloop);
  busy := false;
end;

function TsvSoundData.isplaying : boolean;
begin
  isplaying := data.flags and snd_play <> 0;
end;

function TsvSoundData.ispaused : boolean;
begin
  ispaused := data.flags and snd_pause <> 0;
end;

function TsvSoundData.ismuted : boolean;
begin
  ismuted := data.flags and snd_mute <> 0;
end;

procedure TsvSoundData.mute(active : boolean);
begin
  if (active) then
    data.flags := data.flags or (snd_mute)
    else data.flags := data.flags and not (snd_mute);
end;

procedure TsvSoundData.setspeed(speed : real);
begin
  if not initialized then exit;
  with data do
  begin
    freq := round(ofreq*speed);
    if (freq <> 0) then
    xinc := round(freq/ofreq*65536*ofreq/father^.dat.mainfreq) else xinc := 0;
  end;
end;

procedure TsvSoundData.setfrequency(freq : longword);
begin
  if not initialized then exit;
  data.freq := freq;
  if (freq <> 0)and(data.ofreq <> 0)and(father^.dat.mainfreq <> 0) then
  with data do
  xinc := round(freq/ofreq*65536*ofreq/father^.dat.mainfreq) else data.xinc := 0;
end;

procedure TsvSoundData.setinterpolation(active : boolean);
begin
  with data do
  if (active) then flags := flags or snd_intpol else flags := flags and not snd_intpol;
end;

procedure TsvSoundData.setramping(active : boolean);
begin
  with data do
  if (active) then flags := flags or snd_ramping else flags := flags and not snd_ramping;
end;

procedure TsvSoundData.requestblock(p : pointer;samples : longword);
begin
end;

function TsvSoundData.requestblock_direct16(p : pointer;samples : longword) : longword;
begin
  requestblock_direct16 := 0;
end;

function TsvSoundData.getvu : byte;
begin
  if (data.vucount > 0) then
    data.vuval := (data.vuval div data.vucount) div 48 else data.vuval := data.lastvu;
  if (data.vuval > 255) then data.vuval := 255;

  if (data.flags and snd_mute <> 0) then data.vuval := 0;

  data.lastvu := byte(data.vuval);
  getvu       := data.lastvu;

  data.vuval   := 0;
  data.vucount := 0;
end;
{ --- }
{ TSound Object procedures }
{ --- }
constructor TsvSoundObject.init(var dest : TsvSoundDriver;channels : longword);
var
  i        : integer;
  newchain : psoundchain;

begin
  while busy do;
  busy := true;
  output := addr(dest);
  new(newchain);
  newchain^.next     := output^.chain.next;
  newchain^.current  := addr(self);
  output^.chain.next := newchain;
  inc(output^.data.listcount);
  { Object initialization }
  dat.samplesize  := 2;
  if (output^.data.flags and snd_stereo <> 0) then inc(dat.samplesize,dat.samplesize);

  dat.samplecount := 0;
  dat.remain      := 0;
  dat.mainfreq    := output^.data.freq;
  dat.playchnls   := channels;

  setbuffersize(svSound_DefaultBufferSize);

  for i := 0 to output^.data.inchannels do setvolume(i,256);
  chain.next := nil;
  initialized := true;
  output^.recalcvol;
  busy := false;
end;

destructor TsvSoundObject.done;
var
  i      : longint;
  ch,ch2 : psoundchain;

begin
  if not initialized then exit;
  while busy do;
  busy := true;
  ch  := output^.chain.next;
  ch2 := addr(output^.chain);
  for i := 0 to output^.data.listcount-1 do
  begin
    if (ch^.current = addr(self)) then
    begin
      ch2^.next := ch^.next;
      dispose(ch);
      break;
    end;
    ch2 := ch;
    ch  := ch^.next;
  end;
  dec(output^.data.listcount);

  initialized := false;
  busy := false;
end;

procedure TsvSoundObject.requestblock(p : pointer;samples : longword);
var
  i          : longint;
  ch         : pdatchain;
  count,size : longword;

begin
  if not initialized then exit;
  count := samples;
  while (count > 0) do
  begin
    if (samples > bfsize) then
    begin
      if (dat.remain > 0) then
      begin
        size := dat.remain;
        dat.remain := 0;
      end
      else
        if (count > bfsize) then size := bfsize else size := count;
    end else
    begin
      if (dat.remain > 0) then
      begin
        if (dat.remain < samples) then
          size := dat.remain else size := samples;
        dec(dat.remain,size);
      end
      else size := count;
    end;

    dec(count,size);
    ch := chain.next;
    for i := 0 to dat.listcount-1 do
    begin
      if assigned(ch^.current) then
      ch^.current^.requestblock(p,size);
      ch := ch^.next;
    end;
    inc(dat.samplecount,size);
    if (dat.samplecount mod bfsize = 0) then callproc;
    inc(longword(p),size*output^.data.inchannels*4);
  end;
  dat.remain := bfsize-dat.samplecount mod bfsize;
end;

procedure TsvSoundObject.setbuffersize(samples : longword);
begin
  bfsize          := samples;
  dat.size        := bfsize*output^.data.inchannels*4;
  dat.samplecount := 0;
  dat.remain      := 0;
end;

procedure TsvSoundObject.setchannels(channels : longword);
var
  i : integer;

begin
  dat.playchnls := channels;
  for i := 0 to output^.data.inchannels do setvolume(i,dat.vol[i]);
end;

procedure TsvSoundObject.setvolume(channel : word;volume : longword);
var
  i  : longint;
  ch : pdatchain;

begin
  if (channel > output^.data.outchannels) then exit;
  if (volume > 512) then volume := 512;
  dat.vol[channel] := volume;
  ch := chain.next;

  if (dat.listcount > 0) then
  for i := 0 to dat.listcount-1 do
  begin
    ch^.current^.setvolume(channel,ch^.current^.data.vol[channel]);
    ch := ch^.next;
  end;
end;

function TsvSoundObject.getvolume(channel : word) : word;
begin
  if (channel > output^.data.inchannels) then
    getvolume := 0 else getvolume := dat.vol[channel];
end;

function TsvSoundObject.change : boolean;
begin
  change := (dat.lastcount <> dat.samplecount);
  dat.lastcount := dat.samplecount;
end;

function TsvSoundObject.getv(chn,vol : longword) : longword;
var
  v : longword;

begin
  v := 0;
  if (output^.data.flags and snd_prologic <> 0) then
  begin
    if (chn <= 1) then
      v := (vol * longword(dat.vol[chn])) div dat.playchnls div output^.data.listcount div 4
    else if (chn = 2) then v := round(vol*(dat.vol[chn]/dat.playchnls/output^.data.listcount/4)*0.711); {-3db for S channel}
  end else
  begin
    if (chn < output^.data.inchannels) then
      v := (vol * longword(dat.vol[chn])) div dat.playchnls div output^.data.listcount div 4;
  end;
  getv := v;
end;

procedure TsvSoundObject.callproc;
begin
end;

procedure TsvSoundObject.setinterpolation(active : boolean);
var
  ch   : pdatchain;
  i    : longint;

begin
  if not initialized then exit;
  ch := chain.next;
  for i := 0 to dat.listcount-1 do with ch^.current^.data do
  begin
    if (active) then flags := flags or snd_intpol else flags := flags and not snd_intpol;
    ch := ch^.next;
  end;
end;

procedure TsvSoundObject.setramping(active : boolean);
var
  ch   : pdatchain;
  i    : integer;

begin
  if not initialized then exit;
  ch := chain.next;
  for i := 0 to dat.listcount-1 do with ch^.current^.data do
  begin
    if (active) then flags := flags or snd_ramping else flags := flags and not snd_ramping;
    ch := ch^.next;
  end;
end;

{ --- }


{ TSample Object procedures }
{ --- }
constructor tsample.init(var dest : TsvSoundObject;var source : TsvSoundData);
var
  i : integer;

begin
  inherited init(dest);
  dat.size  := 2*source.getnumofsamples;
  dat.stop  := dat.size shr 1;
  dat.loopstart := noloop;
  data.ofreq := source.getsamplefrequency;
  data.flags := 0;
  data.pos   := 0;
  getmem(data.buffer,dat.size);
  if (data.buffer = nil) then fail;
  source.play;
  source.requestblock_direct16(data.buffer,source.getnumofsamples);
  setfrequency(data.ofreq);
  data.samplesize := 2;
  for i := 0 to 1 do setvolume(i,256);
  for i := 2 to svSound_MaxChannels-1 do setvolume(i,0);
end;

destructor tsample.done;
begin
  if not initialized then exit;
  inherited done;
  freemem(data.buffer);
end;

function tsample.getnumofsamples : longword;
begin
  getnumofsamples := dat.size div getsamplesize;
end;

procedure tsample.setloop(strt,stp : longword);
begin
  if not initialized then exit;

  if (strt > dat.size shr 1)or(stp > dat.size shr 1) then exit;
  if (strt > stp) then exit;
  dat.loopstart := strt;
  data.flags    := data.flags or snd_loop;
  if (stp > 0) then dat.stop := stp else
  begin
    dat.stop      := dat.size shr 1;
    dat.loopstart := noloop;
  end;
end;

procedure tsample.setpos(pos : longword);
var
  i : integer;

begin
  if (pos > dat.stop) then pos := dat.stop;
  data.pos    := pos;
  data.subpos := 0;

  if (pos > dat.loopstart)and(pos < dat.stop) then
  data.flags := data.flags or snd_inloop
  else data.flags := data.flags and not snd_inloop;

  if (data.flags and snd_ramping <> 0) then with data do
  begin
    flags   := flags or snd_inramp;

    for i := 0 to svSound_MaxChannels-1 do
    begin
      oldvol[i] := 0;
      ramppos[i] := 0;
      rampstart[i] := v[i] div 2; { This should be ok ... }
      rampfinal[i] := v[i];
    end;
  end;
end;

function tsample.getpos : longword;
begin
  getpos := data.pos;
end;

function tsample.getsize : longword;
begin
  getsize := dat.size;
end;


procedure tsample.requestblock(p : pointer;samples : longword);
var
  vu        : longint;
  vl,tmp    : tv;
  l,rc      : longint;
  w         : longword;
  val,val2  : longint;
  chnls,i,j : integer;
  newpos    : longword;

begin
  if (data.buffer = nil)or(not initialized)or(samples = 0) then exit;
  while busy do;
  busy  := true;
  val   := 0;
  vu    := 0;
  if (data.flags and snd_mute = 0) then
    vl    := data.v
    else fillchar(vl,sizeof(vl),0);

  if (data.flags and snd_play <> 0) then
  begin
    { ----- win32/FPC ----- }
    chnls := father^.output^.data.inchannels-1;

    if (data.flags and snd_intpol <> 0) then
    { ----- Mix sample data interpolated ----- }
    with data do
    begin
      if (xinc < $10000) then { do upsampling }
      for l := 0 to samples-1 do
      begin

        if (pos < dat.stop-1) then { Sample not at end... }
        begin
          val  := longint(pointer(longword(buffer)+longword(pos) shl 1)^);
          val2 := smallint(val shr 16);
          val  := smallint(val and $FFFF);
          val  := val+((val2-val)*longint(subpos shr 1)) div $8000;
        end

        else { Sample end reached }

        begin
          if (dat.loopstart <> noloop) then
          begin
            { We actually have the first int at buffer end, the second at loop start }
            if (pos >= dat.stop) then
            begin
              pos  := dat.loopstart;
              val  := longint(pointer(longword(buffer)+longword(pos) shl 1)^);
              val2 := smallint(val shr 16);
              val  := smallint(val and $FFFF);
            end
            else
            begin
              val   := smallint(pointer(longword(buffer)+longword(pos) shl 1)^);
              val2  := smallint(pointer(longword(buffer)+longword(dat.loopstart) shl 1)^);
            end;

            flags := flags or snd_inloop;
            val   := val+((val2-val)*(longint(subpos) shr 1)) div $8000;
          end

          else
          //if (pos >= dat.stop) then
          { No looping, stop sample }
          { We actually 'forget' to play the last sample here }
          begin
            flags := flags and not snd_play;
            break;
          end;
        end;

        { Increase position }
        inc(subpos,xinc);
        inc(pos,subpos shr 16);
        subpos := subpos and $FFFF;


        if (flags and snd_inramp = 0) then
        { Mix to every single channel}
        for i := 0 to chnls do
        begin
          if (vl[i] > 0) then inc(longint(p^),val*vl[i]);
          inc(longword(p),4);

        end
        else { Do volume ramping }
        begin
          rc := 0;
          for i := 0 to chnls do
          begin
            if (ramppos[i] < rampsize) then
            begin
              tmp[i] := (rampstart[i]+(rampfinal[i]-rampstart[i])*(ramppos[i]+1) div rampsize);
              inc(longint(p^),val*tmp[i]);
              inc(ramppos[i]);
              inc(rc);
            end else
            if (vl[i] > 0) then inc(longint(p^),val*vl[i]);
            inc(longword(p),4);
          end;
          if (rc = 0) then flags := flags and not snd_inramp;
        end;

       inc(vu,abs(val) shr 8);
      end

      else  { --- do downsampling --- }
      for l := 0 to samples-1 do
      begin
        if (pos+(subpos+xinc) shr 16 < dat.stop) then { normal interpolation... }
        begin
          newpos := pos+(subpos+xinc) shr 16;
          val := 0;
          for w := pos to newpos do inc(val,smallint(pointer(longword(buffer)+w shl 1)^));
          val := val div longint(newpos-pos+1);
        end

        else { Overflow... }

        begin
          if (dat.loopstart <> noloop) then
          begin
            { We actually have the first int at buffer end, the others at loop start }
            val := 0;
            for w := pos to dat.stop do inc(val,smallint(pointer(longword(buffer)+w shl 1)^));
            j := dat.stop-pos+1;

            pos    := dat.loopstart;
            newpos := dat.loopstart+(subpos+xinc) shr 16{-j}; {!!}

            for w := pos to newpos-1 do inc(val,smallint(pointer(longword(buffer)+w shl 1)^));

            val   := val div longint(newpos-pos+1+longword(j));
            flags := flags or snd_inloop;
          end

          else   { No looping, stop sample }
          begin
            //val  := smallint(pointer(longword(buffer)+pos shl 1)^);
            flags := flags and not snd_play;
            break;
          end;
        end;

        { Increase position }
        inc(subpos,xinc);
        inc(pos,subpos shr 16);
        subpos := subpos and $FFFF;
        if (flags and snd_inramp = 0) then
        { Mix to every single channel}
        for i := 0 to chnls do
        begin
          if (vl[i] > 0) then inc(longint(p^),val*vl[i]);
          inc(longword(p),4);
        end
        else { Do volume ramping }
        begin
          rc := 0;
          for i := 0 to chnls do
          begin
            if (ramppos[i] < rampsize) then
            begin
              tmp[i] := (rampstart[i]+(rampfinal[i]-rampstart[i])*(ramppos[i]+1) div rampsize);
              inc(longint(p^),val*tmp[i]);
              inc(ramppos[i]);
              inc(rc);
            end else
            if (vl[i] > 0) then inc(longint(p^),val*vl[i]);
            inc(longword(p),4);
          end;
          if (rc = 0) then flags := flags and not snd_inramp;
        end;
        inc(vu,abs(val) shr 8);
      end;
    end

    else
    { ----- Normal sample data mixing ----- }
    for l := 0 to samples-1 do
    with data do
    begin

      if (pos >= dat.stop) then
      begin
        if (dat.loopstart <> noloop) then
        begin
          pos := dat.loopstart;
          flags := flags or snd_inloop;
        end
        else   { No looping, stop sample }
        begin
          flags := flags and not snd_play;
          break;
        end;
      end;

      val := smallint(pointer(longword(buffer)+pos shl 1)^);
      if (flags and snd_inramp = 0) then
      { Mix to every single channel}
      for i := 0 to chnls do
      begin
        if (vl[i] > 0) then inc(longint(p^),val*vl[i]);
        inc(longword(p),4);
      end
      else { Do volume ramping }
      begin
        rc := 0;
        for i := 0 to chnls do
        begin
          if (ramppos[i] < rampsize) then
          begin
            tmp[i] := (rampstart[i]+(rampfinal[i]-rampstart[i])*(ramppos[i]+1) div rampsize);
            inc(longint(p^),val*tmp[i]);
            inc(ramppos[i]);
            inc(rc);
          end else
          if (vl[i] > 0) then inc(longint(p^),val*vl[i]);
          inc(longword(p),4);
        end;
        if (rc = 0) then flags := flags and not snd_inramp;
      end;

      inc(vu,abs(val) shr 8);

      inc(subpos,xinc);
      inc(pos,subpos shr 16);
      subpos := subpos and $FFFF;
    end;
  end;
  if (data.vol[0] > data.vol[1]) then
    inc(data.vuval,vu*(data.vol[0])) else
    inc(data.vuval,vu*(data.vol[1]));
  inc(data.vucount,samples);
  busy := false;
end;
{ --- }


{ TChannel object procedures }
{ --- }
constructor tchannel.init(var dest : TsvSoundObject);
begin
  TsvSoundData.init(dest);
  data.samplesize := 2;
end;

destructor tchannel.done;
begin
  TsvSoundData.done;
end;

procedure tchannel.setsample(s : pointer;size,freq : longword);
var
  i : integer;

begin
  if (s = nil) then exit;

  data.buffer   := s;
  dat.size      := size;

  data.ofreq    := freq;
  data.flags    := data.flags and not (snd_command or snd_loopdata);
  data.pos      := 0;
  dat.loopstart := noloop;
  dat.stop      := size shr 1;

  setspeed(1);
  for i := 0 to 1 do setvolume(i,256);
  for i := 2 to svSound_MaxChannels-1 do setvolume(i,0);
end;
{ --- }
end.
